<?php
/**
 * Category Model
 * Model untuk mengelola data master kategori pengaduan
 */

class Category {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Get all categories
     */
    public function getAll($activeOnly = false) {
        $sql = "SELECT * FROM categories";
        
        if ($activeOnly) {
            $sql .= " WHERE is_active = 1";
        }
        
        $sql .= " ORDER BY name ASC";
        
        return $this->db->fetchAll($sql);
    }

    /**
     * Get active categories for public form
     */
    public function getActive() {
        return $this->getAll(true);
    }

    /**
     * Find category by ID
     */
    public function findById($id) {
        $sql = "SELECT * FROM categories WHERE id = ?";
        return $this->db->fetch($sql, [$id]);
    }

    /**
     * Check if category name already exists
     */
    public function existsByName($name, $excludeId = null) {
        $sql = "SELECT COUNT(*) as count FROM categories WHERE name = ?";
        $params = [$name];
        
        if ($excludeId) {
            $sql .= " AND id != ?";
            $params[] = $excludeId;
        }
        
        $result = $this->db->fetch($sql, $params);
        return $result['count'] > 0;
    }

    /**
     * Create new category
     */
    public function create($data) {
        // Check if name already exists
        if ($this->existsByName($data['name'])) {
            throw new Exception("Kategori dengan nama tersebut sudah ada");
        }

        $sql = "INSERT INTO categories (name, description, is_active) 
                VALUES (?, ?, ?)";
        
        $params = [
            $data['name'],
            $data['description'] ?? null,
            isset($data['is_active']) ? 1 : 0
        ];

        $this->db->query($sql, $params);
        return $this->db->lastInsertId();
    }

    /**
     * Update category
     */
    public function update($id, $data) {
        // Check if name already exists (excluding current record)
        if ($this->existsByName($data['name'], $id)) {
            throw new Exception("Kategori dengan nama tersebut sudah ada");
        }

        $sql = "UPDATE categories 
                SET name = ?, 
                    description = ?, 
                    is_active = ?,
                    updated_at = NOW()
                WHERE id = ?";
        
        $params = [
            $data['name'],
            $data['description'] ?? null,
            isset($data['is_active']) ? 1 : 0,
            $id
        ];

        $this->db->query($sql, $params);
    }

    /**
     * Delete category
     */
    public function delete($id) {
        // Check if category is being used in tickets
        $sql = "SELECT COUNT(*) as count FROM tickets WHERE category = (SELECT name FROM categories WHERE id = ?)";
        $result = $this->db->fetch($sql, [$id]);
        
        if ($result['count'] > 0) {
            throw new Exception("Kategori tidak dapat dihapus karena sedang digunakan pada " . $result['count'] . " pengaduan");
        }

        $sql = "DELETE FROM categories WHERE id = ?";
        $this->db->query($sql, [$id]);
    }

    /**
     * Toggle active status
     */
    public function toggleActive($id) {
        $sql = "UPDATE categories 
                SET is_active = NOT is_active,
                    updated_at = NOW()
                WHERE id = ?";
        $this->db->query($sql, [$id]);
    }

    /**
     * Get statistics
     */
    public function getStats() {
        $sql = "SELECT 
                    COUNT(*) as total,
                    SUM(CASE WHEN is_active = 1 THEN 1 ELSE 0 END) as active,
                    SUM(CASE WHEN is_active = 0 THEN 1 ELSE 0 END) as inactive
                FROM categories";
        
        return $this->db->fetch($sql);
    }

    /**
     * Get category usage count
     */
    public function getUsageCount($categoryName) {
        $sql = "SELECT COUNT(*) as count FROM tickets WHERE category = ?";
        $result = $this->db->fetch($sql, [$categoryName]);
        return $result['count'];
    }
}

