<?php
/**
 * File Upload Helper - Secure File Upload Handler
 * Mencegah serangan double extension, MIME type spoofing, dll
 */

class FileUpload {
    /**
     * Upload file dengan validasi keamanan lengkap
     * 
     * @param array $file $_FILES array
     * @param string $uploadDir Direktori upload
     * @param array $allowedMimeTypes Array MIME types yang diizinkan
     * @param int $maxSize Ukuran maksimal dalam bytes
     * @param string $filenamePrefix Prefix untuk nama file
     * @return array ['success' => bool, 'filename' => string|null, 'mime_type' => string|null, 'error' => string|null]
     */
    public static function uploadSecure($file, $uploadDir, $allowedMimeTypes, $maxSize = 5242880, $filenamePrefix = 'file') {
        // Validasi file ada
        if (!isset($file) || $file['error'] === UPLOAD_ERR_NO_FILE) {
            return ['success' => false, 'filename' => null, 'mime_type' => null, 'error' => 'File tidak ditemukan'];
        }

        // Validasi error upload
        if ($file['error'] !== UPLOAD_ERR_OK) {
            return ['success' => false, 'filename' => null, 'mime_type' => null, 'error' => 'Gagal mengupload file. Error code: ' . $file['error']];
        }

        // Validasi ukuran file
        if ($file['size'] > $maxSize) {
            return ['success' => false, 'filename' => null, 'mime_type' => null, 'error' => 'Ukuran file terlalu besar. Maksimal ' . ($maxSize / 1024 / 1024) . 'MB'];
        }

        // Validasi direktori upload
        if (!is_dir($uploadDir) || !is_writable($uploadDir)) {
            return ['success' => false, 'filename' => null, 'mime_type' => null, 'error' => 'Folder upload tidak tersedia atau tidak dapat ditulis'];
        }

        // Deteksi MIME type menggunakan finfo (PENTING: jangan percaya $_FILES['type'])
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mimeType = finfo_file($finfo, $file['tmp_name']);
        finfo_close($finfo);

        // Validasi MIME type
        if (!in_array($mimeType, $allowedMimeTypes)) {
            return ['success' => false, 'filename' => null, 'mime_type' => null, 'error' => 'Tipe file tidak diizinkan. Hanya ' . implode(', ', $allowedMimeTypes)];
        }

        // Map MIME type ke ekstensi yang aman (PENTING: untuk mencegah double extension attack)
        $mimeToExt = [
            'image/jpeg' => 'jpg',
            'image/png' => 'png',
            'image/gif' => 'gif',
            'image/webp' => 'webp',
            'application/pdf' => 'pdf',
            'application/msword' => 'doc',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document' => 'docx',
            'application/vnd.ms-excel' => 'xls',
            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet' => 'xlsx'
        ];

        // Generate ekstensi dari MIME type yang sudah divalidasi (BUKAN dari nama file!)
        // Ini mencegah serangan double extension seperti shell.php.png
        if (!isset($mimeToExt[$mimeType])) {
            return ['success' => false, 'filename' => null, 'mime_type' => null, 'error' => 'Tipe file tidak valid atau tidak didukung'];
        }

        $extension = $mimeToExt[$mimeType]; // ← AMAN: ekstensi dari MIME type, bukan nama file

        // Generate nama file dengan ekstensi yang aman
        $filename = $filenamePrefix . '_' . time() . '_' . uniqid() . '.' . $extension;
        $filepath = rtrim($uploadDir, '/') . '/' . $filename;

        // Pindahkan file
        if (!move_uploaded_file($file['tmp_name'], $filepath)) {
            return ['success' => false, 'filename' => null, 'mime_type' => null, 'error' => 'Tidak dapat menyimpan file'];
        }

        return [
            'success' => true,
            'filename' => $filename,
            'mime_type' => $mimeType,
            'error' => null
        ];
    }

    /**
     * Upload file gambar dengan validasi khusus untuk gambar
     * 
     * @param array $file $_FILES array
     * @param string $uploadDir Direktori upload
     * @param int $maxSize Ukuran maksimal dalam bytes (default: 5MB)
     * @param string $filenamePrefix Prefix untuk nama file
     * @return array ['success' => bool, 'filename' => string|null, 'mime_type' => string|null, 'error' => string|null]
     */
    public static function uploadImage($file, $uploadDir, $maxSize = 5242880, $filenamePrefix = 'image') {
        $allowedMimeTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
        return self::uploadSecure($file, $uploadDir, $allowedMimeTypes, $maxSize, $filenamePrefix);
    }
}

