<?php
/**
 * Category Controller
 * Controller untuk mengelola master kategori pengaduan
 */

class CategoryController {
    private $categoryModel;

    public function __construct() {
        $this->categoryModel = new Category();
    }

    /**
     * Display list of categories
     */
    public function index() {
        Middleware::role('admin');
        $user = Auth::user();
        
        $categories = $this->categoryModel->getAll();
        $stats = $this->categoryModel->getStats();
        
        // Get usage count for each category
        foreach ($categories as &$category) {
            $category['usage_count'] = $this->categoryModel->getUsageCount($category['name']);
        }
        unset($category);
        
        require __DIR__ . '/../views/categories/index.php';
    }

    /**
     * Show create form
     */
    public function create() {
        Middleware::role('admin');
        $user = Auth::user();
        
        require __DIR__ . '/../views/categories/form.php';
    }

    /**
     * Store new category
     */
    public function store() {
        Middleware::role('admin');
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /halopemko/public/index.php?action=categories');
            exit;
        }

        $name = trim($_POST['name'] ?? '');
        $description = trim($_POST['description'] ?? '');
        $is_active = isset($_POST['is_active']);

        // Validation
        if (empty($name)) {
            $_SESSION['error'] = "Nama kategori wajib diisi";
            header('Location: /halopemko/public/index.php?action=category_create');
            exit;
        }

        try {
            $data = [
                'name' => $name,
                'description' => $description,
                'is_active' => $is_active
            ];

            $this->categoryModel->create($data);
            $_SESSION['success'] = "Kategori berhasil ditambahkan";
            header('Location: /halopemko/public/index.php?action=categories');
            exit;
        } catch (Exception $e) {
            $_SESSION['error'] = $e->getMessage();
            header('Location: /halopemko/public/index.php?action=category_create');
            exit;
        }
    }

    /**
     * Show edit form
     */
    public function edit() {
        Middleware::role('admin');
        $user = Auth::user();
        
        $id = $_GET['id'] ?? null;
        
        if (!$id) {
            $_SESSION['error'] = "ID kategori tidak valid";
            header('Location: /halopemko/public/index.php?action=categories');
            exit;
        }

        $category = $this->categoryModel->findById($id);
        
        if (!$category) {
            $_SESSION['error'] = "Kategori tidak ditemukan";
            header('Location: /halopemko/public/index.php?action=categories');
            exit;
        }

        require __DIR__ . '/../views/categories/form.php';
    }

    /**
     * Update category
     */
    public function update() {
        Middleware::role('admin');
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /halopemko/public/index.php?action=categories');
            exit;
        }

        $id = $_POST['id'] ?? null;
        $name = trim($_POST['name'] ?? '');
        $description = trim($_POST['description'] ?? '');
        $is_active = isset($_POST['is_active']);

        if (!$id) {
            $_SESSION['error'] = "ID kategori tidak valid";
            header('Location: /halopemko/public/index.php?action=categories');
            exit;
        }

        // Validation
        if (empty($name)) {
            $_SESSION['error'] = "Nama kategori wajib diisi";
            header('Location: /halopemko/public/index.php?action=category_edit&id=' . $id);
            exit;
        }

        try {
            $data = [
                'name' => $name,
                'description' => $description,
                'is_active' => $is_active
            ];

            $this->categoryModel->update($id, $data);
            $_SESSION['success'] = "Kategori berhasil diupdate";
            header('Location: /halopemko/public/index.php?action=categories');
            exit;
        } catch (Exception $e) {
            $_SESSION['error'] = $e->getMessage();
            header('Location: /halopemko/public/index.php?action=category_edit&id=' . $id);
            exit;
        }
    }

    /**
     * Delete category
     */
    public function delete() {
        Middleware::role('admin');
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /halopemko/public/index.php?action=categories');
            exit;
        }

        $id = $_POST['id'] ?? null;

        if (!$id) {
            $_SESSION['error'] = "ID kategori tidak valid";
            header('Location: /halopemko/public/index.php?action=categories');
            exit;
        }

        $_SESSION['error'] = "Penghapusan permanen kategori dinonaktifkan. Silakan nonaktifkan kategori untuk menyembunyikannya.";

        header('Location: /halopemko/public/index.php?action=categories');
        exit;
    }

    /**
     * Toggle active status
     */
    public function toggleActive() {
        Middleware::role('admin');
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /halopemko/public/index.php?action=categories');
            exit;
        }

        $id = $_POST['id'] ?? null;

        if (!$id) {
            $_SESSION['error'] = "ID kategori tidak valid";
            header('Location: /halopemko/public/index.php?action=categories');
            exit;
        }

        try {
            $this->categoryModel->toggleActive($id);
            $_SESSION['success'] = "Status kategori berhasil diubah";
        } catch (Exception $e) {
            $_SESSION['error'] = $e->getMessage();
        }

        header('Location: /halopemko/public/index.php?action=categories');
        exit;
    }
}

